<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\NotificationLog;
use App\Models\SupportTicket;
use App\Models\Transaction;
use App\Models\User;
use App\Models\Withdrawal;
use App\Models\Invest;
use Illuminate\Http\Request;
use App\Models\GeneralSetting;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;


class ManageUsersController extends Controller
{


   public function investedUsers()
    {
        $pageTitle = 'Invested Users';
        $invested     = Invest::orderBy('id', 'desc')->pluck('user_id');
         
        $users     = User::whereIn('id', $invested)->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.users.list', compact('pageTitle', 'users'));
    }
    
    public function allUsers()
    {
        $pageTitle = 'All Users';
        $users     = $this->userData();
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function activeUsers()
    {
        $pageTitle = 'Active Users';
        $users     = $this->userData('active');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function bannedUsers()
    {
        $pageTitle = 'Banned Users';
        $users     = $this->userData('banned');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function emailUnverifiedUsers()
    {
        $pageTitle = 'Email Unverified Users';
        $users     = $this->userData('emailUnverified');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function kycUnverifiedUsers()
    {
        $pageTitle = 'KYC Unverified Users';
        $users     = $this->userData('kycUnverified');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function kycPendingUsers()
    {
        $pageTitle = 'KYC Unverified Users';
        $users     = $this->userData('kycPending');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function emailVerifiedUsers()
    {
        $pageTitle = 'Email Verified Users';
        $users     = $this->userData('emailVerified');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function mobileUnverifiedUsers()
    {
        $pageTitle = 'Mobile Unverified Users';
        $users     = $this->userData('mobileUnverified');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function mobileVerifiedUsers()
    {
        $pageTitle = 'Mobile Verified Users';
        $users     = $this->userData('mobileVerified');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    public function usersWithBalance()
    {
        $pageTitle = 'Users with Balance';
        $users     = $this->userData('withBalance');
        return view('admin.users.list', compact('pageTitle', 'users'));
    }

    protected function userData($scope = null)
    {
        if ($scope) {
            $users = User::$scope();
        } else {
            $users = User::query();
        }

        return $users->searchable(['username', 'email'])->orderBy('id', 'desc')->paginate(getPaginate());
    }

    public function detail($id)
    {
        $user      = User::findOrFail($id);
        $pageTitle = 'User Detail - ' . $user->username;

        $data['totalInvest'] = Invest::where('user_id', $user->id)->sum('amount');
         $data['referral_earnings'] = Transaction::where('remark', 'referral_commission')->where('user_id', $user->id)->sum('amount');
  
        $totalDeposit     = Deposit::where('user_id', $user->id)->where('status', 1)->sum('amount');
        $totalWithdrawals = Withdrawal::where('user_id', $user->id)->where('status', 1)->sum('amount');
        $totalTransaction = Transaction::where('user_id', $user->id)->count();
        $pendingTicket    = SupportTicket::where('user_id', $user->id)->whereIN('status', [0, 2])->count();
        $countries        = json_decode(file_get_contents(resource_path('views/partials/country.json')));
        return view('admin.users.detail', compact('pageTitle', 'user', 'totalDeposit', 'totalWithdrawals', 'totalTransaction', 'pendingTicket', 'countries'),$data);
    }

    public function kycDetails($id)
    {
        $pageTitle = 'KYC Details';
        $user      = User::findOrFail($id);
        return view('admin.users.kyc_detail', compact('pageTitle', 'user'));
    }

    public function kycApprove($id)
    {
        $user     = User::findOrFail($id);
        $user->kv = 1;
        $user->save();

        notify($user, 'KYC_APPROVE', []);

        $notify[] = ['success', 'KYC approved successfully'];
        return to_route('admin.users.kyc.pending')->withNotify($notify);
    }

    public function kycReject($id)
    {
        $user = User::findOrFail($id);
        foreach ($user->kyc_data as $kycData) {
            if ($kycData->type == 'file') {
                fileManager()->removeFile(getFilePath('verify') . '/' . $kycData->value);
            }
        }
        $user->kv       = 0;
        $user->kyc_data = null;
        $user->save();

        notify($user, 'KYC_REJECT', []);

        $notify[] = ['success', 'KYC rejected successfully'];
        return to_route('admin.users.kyc.pending')->withNotify($notify);
    }

    public function update(Request $request, $id)
    {
        $user         = User::findOrFail($id);


        $request->validate([
            'firstname' => 'required|string|max:40',
            'lastname'  => 'required|string|max:40',
            // NEW: wallet field from the view
            'wallet'    => 'required|string|max:100',
            // If you want stricter ETH/Polygon format, use:
            // 'wallet' => ['required','regex:/^0x[a-fA-F0-9]{40}$/'],
            
        ]);
        
        
        
        
        $user->firstname    = $request->firstname;
        $user->lastname     = $request->lastname;

        $user->ev = $request->ev ? 1 : 0;
        $user->sv = $request->sv ? 1 : 0;
        $user->ts = $request->ts ? 1 : 0;
        
        
        // NEW: persist wallet
        $user->wallet = trim($request->wallet);
    
    
    
        if (!$request->kv) {
            $user->kv = 0;
            if ($user->kyc_data) {
                foreach ($user->kyc_data as $kycData) {
                    if ($kycData->type == 'file') {
                        fileManager()->removeFile(getFilePath('verify') . '/' . $kycData->value);
                    }
                }
            }
            $user->kyc_data = null;
        } else {
            $user->kv = 1;
        }
        
        $user->save();

        $notify[] = ['success', 'User details updated successfully'];
        return back()->withNotify($notify);
    }
    
    public function addSubBalance(Request $request, $id)
    {
        $request->validate([
            'amount'      => 'required|numeric|gt:0',
            'act'         => 'required|in:add,sub',
            'wallet_type' => 'required|in:token_balance',
            'remark'      => 'required|string|max:255',
        ]);
    
        $user        = User::findOrFail($id);
        $amount      = (float)$request->amount;
        $wallet      = $request->wallet_type; // "token_balance"
        $trx         = getTrx();
        $notify      = [];
    
        // Load general settings (for deposit_commission toggle)
        $general = GeneralSetting::first();
    
        DB::transaction(function () use (
            $request, $user, $amount, $wallet, $trx, &$notify, $general
        ) {
            $transaction              = new Transaction();
            $transaction->user_id     = $user->id;
            $transaction->amount      = $amount;
            $transaction->charge      = 0;
            $transaction->trx         = $trx;
            $transaction->details     = $request->remark;
            $transaction->wallet_type = $wallet;
    
            if ($request->act === 'add') {
                $user->$wallet = (float)$user->$wallet + $amount;
    
                $transaction->trx_type = '+';
                $transaction->remark   = 'balance_add';
    
                $notify[] = ['success', gs('cur_sym') . $amount . ' added successfully'];
                $notifyTemplate = 'BAL_ADD';
            } else {
                if ($amount > (float)$user->$wallet) {
                    // throwing inside transaction will rollback
                    throw new \RuntimeException($user->username . ' doesn\'t have sufficient balance.');
                }
                $user->$wallet = (float)$user->$wallet - $amount;
    
                $transaction->trx_type = '-';
                $transaction->remark   = 'balance_subtract';
    
                $notify[] = ['success', gs('cur_sym') . $amount . ' subtracted successfully'];
                $notifyTemplate = 'BAL_SUB';
            }
    
            $user->save();
    
            // finalize transaction row with post balance
            $transaction->post_balance = $user->$wallet;
            $transaction->save();
    
            // Notify the user about the direct balance change
            notify($user, $notifyTemplate, [
                'trx'          => $trx,
                'amount'       => showAmount($amount),
                'remark'       => $request->remark,
                'post_balance' => showAmount($user->$wallet),
            ]);
    
            /**
             * Commission hook:
             * If admin ADDS to token_balance, give level commissions using the
             * "deposit_commission" scheme (as discussed previously).
             */
            if (
                $request->act === 'add' &&
                $wallet === 'token_balance' &&
                $amount > 0 &&
                $general && (int)$general->deposit_commission === 1
            ) {
                // This will respect the token unlock rules on REFERRERS we implemented earlier
                \App\Lib\HyipLab::levelCommission(
                    $user,                   // the receiver of the tokens (downline)
                    $amount,                 // commission base amount
                    'deposit_commission',    // commission type / tiers to use
                    $trx,                    // same trx id for traceability
                    $general                 // settings object (kept for API parity)
                );
            }
        });
    
        return back()->withNotify($notify);
    }

    public function addSubBalanceTwo(Request $request, $id)
    {
        $request->validate([
            'amount'      => 'required|numeric|gt:0',
            'act'         => 'required|in:add,sub',
            'wallet_type' => 'required|in:token_balance',
            'remark'      => 'required|string|max:255',
        ]);

        $user   = User::findOrFail($id);
        $amount = $request->amount;
        $wallet = $request->wallet_type;
        $trx    = getTrx();

        $transaction = new Transaction();

        if ($request->act == 'add') {
            $user->$wallet += $amount;

            $transaction->trx_type = '+';
            $transaction->remark   = 'balance_add';

            $notifyTemplate = 'BAL_ADD';

            $notify[] = ['success', gs('cur_sym') . $amount . ' added successfully'];

        } else {
            if ($amount > $user->$wallet) {
                $notify[] = ['error', $user->username . ' doesn\'t have sufficient balance.'];
                return back()->withNotify($notify);
            }

            $user->$wallet -= $amount;
            $transaction->trx_type = '-';
            $transaction->remark   = 'balance_subtract';

            $notifyTemplate = 'BAL_SUB';
            $notify[]       = ['success', gs('cur_sym') . $amount . ' subtracted successfully'];
        }

        $user->save();

        $transaction->user_id      = $user->id;
        $transaction->amount       = $amount;
        $transaction->post_balance = $user->$wallet;
        $transaction->charge       = 0;
        $transaction->trx          = $trx;
        $transaction->details      = $request->remark;
        $transaction->wallet_type  = $wallet;
        $transaction->save();

        notify($user, $notifyTemplate, [
            'trx'          => $trx,
            'amount'       => showAmount($amount),
            'remark'       => $request->remark,
            'post_balance' => showAmount($user->$wallet),
        ]);

        return back()->withNotify($notify);
    }

    public function login($id)
    {
        Auth::loginUsingId($id);
        return to_route('user.home');
    }

    public function status(Request $request, $id)
    {
        $user = User::findOrFail($id);
        if ($user->status == 1) {
            $request->validate([
                'reason' => 'required|string|max:255',
            ]);
            $user->status     = 0;
            $user->ban_reason = $request->reason;
            $notify[]         = ['success', 'User banned successfully'];
        } else {
            $user->status     = 1;
            $user->ban_reason = null;
            $notify[]         = ['success', 'User unbanned successfully'];
        }
        $user->save();
        return back()->withNotify($notify);
    }

    public function showNotificationSingleForm($id)
    {
        $user    = User::findOrFail($id);
        if (!gs('en') && !gs('sn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.users.detail', $user->id)->withNotify($notify);
        }
        $pageTitle = 'Send Notification to ' . $user->username;
        return view('admin.users.notification_single', compact('pageTitle', 'user'));
    }

    public function sendNotificationSingle(Request $request, $id)
    {
        $request->validate([
            'message' => 'required|string',
            'subject' => 'required|string',
        ]);

        $user = User::findOrFail($id);
        notify($user, 'DEFAULT', [
            'subject' => $request->subject,
            'message' => $request->message,
        ]);
        $notify[] = ['success', 'Notification sent successfully'];
        return back()->withNotify($notify);
    }

    public function showNotificationAllForm()
    {
        if (!gs('en') && !gs('sn')) {
            $notify[] = ['warning', 'Notification options are disabled currently'];
            return to_route('admin.dashboard')->withNotify($notify);
        }
        $users     = User::active()->count();
        $pageTitle = 'Notification to Verified Users';
        return view('admin.users.notification_all', compact('pageTitle', 'users'));
    }

    public function sendNotificationAll(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start' => 'required',
            'batch' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => $validator->errors()->all()]);
        }

        $users = User::oldest()->active()->skip($request->start)->limit($request->batch)->get();
        foreach ($users as $user) {
            notify($user, 'DEFAULT', [
                'subject' => $request->subject,
                'message' => $request->message,
            ]);
        }

        return response()->json([
            'total_sent' => $users->count(),
        ]);
    }

    public function notificationLog($id)
    {
        $user      = User::findOrFail($id);
        $pageTitle = 'Notifications Sent to ' . $user->username;
        $logs      = NotificationLog::where('user_id', $id)->with('user')->orderBy('id', 'desc')->paginate(getPaginate());
        return view('admin.reports.notification_history', compact('pageTitle', 'logs', 'user'));
    }
}
