<?php

namespace App\Http\Controllers\User\Auth;

use App\Http\Controllers\Controller;
use App\Models\AdminNotification;
use App\Models\User;
use App\Models\UserLogin;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Auth\Events\Registered;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Http\Request;
use kornrunner\Keccak;
use Elliptic\EC;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Illuminate\Support\Facades\Cache;


class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */

    protected $username;

    /**
     * Create a new controller instance.
     *
     * @return void
     */


    public function __construct()
    {
        parent::__construct();
        $this->middleware('guest')->except('logout');
        $this->username = $this->findUsername();
        $this->middleware('auth')->only('updateWalletBalance');
    
    }

    public function showLoginForm()
    {
        $pageTitle = "Login";
        return view($this->activeTemplate. 'user.auth.login', compact('pageTitle'));
    }

    public function login(Request $request)
    {

        $this->validateLogin($request);

        $request->session()->regenerateToken();

        if(!verifyCaptcha()){
            $notify[] = ['error','Invalid captcha provided'];
            return back()->withNotify($notify);
        }

        // If the class is using the ThrottlesLogins trait, we can automatically throttle
        // the login attempts for this application. We'll key this by the username and
        // the IP address of the client making these requests into this application.
        if ($this->hasTooManyLoginAttempts($request)) {
            $this->fireLockoutEvent($request);

            return $this->sendLockoutResponse($request);
        }

        if ($this->attemptLogin($request)) {
            return $this->sendLoginResponse($request);
        }

        // If the login attempt was unsuccessful we will increment the number of attempts
        // to login and redirect the user back to the login form. Of course, when this
        // user surpasses their maximum number of attempts they will get locked out.
        $this->incrementLoginAttempts($request);


        return $this->sendFailedLoginResponse($request);
    }

    public function findUsername()
    {
        $login = request()->input('username');

        $fieldType = filter_var($login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';
        request()->merge([$fieldType => $login]);
        return $fieldType;
    }

    public function username()
    {
        return $this->username;
    }

    protected function validateLogin(Request $request)
    {

        $request->validate([
            $this->username() => 'required|string',
            'password' => 'required|string',
        ]);

    }

  

    public function authenticated(Request $request, $user)
    {
        $user->tv = $user->ts == 1 ? 0 : 1;
        $user->save();
        $ip = getRealIP();
        $exist = UserLogin::where('user_ip',$ip)->first();
        $userLogin = new UserLogin();
        if ($exist) {
            $userLogin->longitude =  $exist->longitude;
            $userLogin->latitude =  $exist->latitude;
            $userLogin->city =  $exist->city;
            $userLogin->country_code = $exist->country_code;
            $userLogin->country =  $exist->country;
        }else{
            $info = json_decode(json_encode(getIpInfo()), true);
            $userLogin->longitude =  @implode(',',$info['long']);
            $userLogin->latitude =  @implode(',',$info['lat']);
            $userLogin->city =  @implode(',',$info['city']);
            $userLogin->country_code = @implode(',',$info['code']);
            $userLogin->country =  @implode(',', $info['country']);
        }

        $userAgent = osBrowser();
        $userLogin->user_id = $user->id;
        $userLogin->user_ip =  $ip;

        $userLogin->browser = @$userAgent['browser'];
        $userLogin->os = @$userAgent['os_platform'];
        $userLogin->save();

        return to_route('user.home');
    }
    
    
     public function message(Request $request)
    {
        $request->validate(['wallet_address' => 'required|string']);
   
        $address = strtolower($request->wallet_address);
        
        // Use a defensive try/catch for unreliable custom helpers gs() and getTrx()
        try {
            $siteName = gs('site_name') ?? 'Your Site Name'; // gs() is an external helper
            $nonce    = strtoupper(getTrx() ?? Str::random(10)); // getTrx() is an external helper
        } catch (Throwable $e) {
            Log::warning('Helper function failed (gs or getTrx) in message()', ['error' => $e->getMessage()]);
            $siteName = 'Your Site Name';
            $nonce    = strtoupper(Str::random(10));
        }
   
        $message = "{$siteName} wants you to sign in with your Polygon account {$address}."
                  . " By sign in i'am agree with {$siteName} privacy & policy."
                  . "\n\nNonce: {$nonce}\nIssued At: " . now();
   
        Cache::put("siwe:{$address}", $nonce, now()->addMinutes(10));
   
        return response()->json([
            'success' => true,
            'address' => $address,
            'nonce'   => $nonce,
            'message' => $message,
        ]);
    }

    /**
     * Verifies the signature and logs in/registers the user.
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function verify(Request $request)
    {
        // 1. Define payload early for logging safety (Fix for "Undefined variable $message")
        $payload = $request->all();

        try {
            // 2. Validation
            $request->validate([
                'wallet_address' => ['required','string'],
                'message'        => ['required','string'],
                'signature'      => ['required','string','regex:/^0x[0-9a-fA-F]{130}$/'],
                'chain_id'       => ['nullable','string'],
            ]);
   
            $address   = strtolower($request->input('wallet_address'));
            $message   = $request->input('message');
            $signature = $request->input('signature');
   
            // 3. Nonce Verification
            if (!preg_match('/Nonce:\s*([A-Z0-9]+)/', $message, $m)) {
                return response()->json(['success' => false, 'message' => 'Nonce not found in message.'], 422);
            }
            $sentNonce = $m[1];
   
            $cachedNonce = Cache::pull("siwe:{$address}");
            if (!$cachedNonce || !hash_equals($cachedNonce, $sentNonce)) {
                return response()->json(['success' => false, 'message' => 'Invalid or expired nonce.'], 422);
            }
   
            // 4. Signature Verification
            if (!$this->verifySignature($message, $signature, $address)) {
                return response()->json(['success' => false, 'message' => 'Signature verification failed.'], 422);
            }
   
            // 5. User Retrieval/Registration
            $user = User::where('wallet', $address)->first();
            $isNewUser = !$user;

            if ($isNewUser) {
                $user = new User();
                $user->wallet       = $address;
                $user->wallet_name  = 'MetaMask';
                $user->message      = $sentNonce; // Consider using username/email if required
                $user->save();
            }
   
            // 6. Login User (Must happen before optional logging to ensure success)
            Auth::login($user, true);

            // --- START OPTIONAL LOGGING AND NOTIFICATIONS ---
            
            // 6a. Admin Notification (Defensively wrapped to catch urlPath() errors)
            if ($isNewUser) {
                try {
                    $adminNotification = new AdminNotification();
                    $adminNotification->user_id   = $user->id;
                    $adminNotification->title     = 'New member registered';
                    // urlPath() is an external helper, use a fallback if it fails
                    $adminNotification->click_url = urlPath('admin.users.detail', $user->id) ?? route('admin.users.detail', $user->id);
                    $adminNotification->save();
                } catch (Throwable $e) {
                    Log::warning('Failed to create Admin Notification for new user.', [
                        'user_id' => $user->id,
                        'error'   => $e->getMessage(),
                    ]);
                }
            }
   
            // 6b. User Login/IP Logging (Defensively wrapped to catch getRealIP, osBrowser, getIpInfo errors)
            try {
                // getRealIP() and osBrowser() are custom helpers and often fail
                $ip     = getRealIP() ?? $request->ip(); // Fallback to Laravel's standard IP
                $exist  = UserLogin::where('user_ip', $ip)->first();
                $userLogin = new UserLogin();
   
                if ($exist) {
                    $userLogin->longitude    = $exist->longitude;
                    $userLogin->latitude     = $exist->latitude;
                    $userLogin->city         = $exist->city;
                    $userLogin->country_code = $exist->country_code;
                    $userLogin->country      = $exist->country;
                } else {
                    // getIpInfo() is a common 500 source due to external API calls
                    $info = json_decode(json_encode(getIpInfo()), true); 
                    
                    // Use @ and null coalescing for maximum protection against array errors
                    $userLogin->longitude    = @implode(',', $info['long'] ?? []);
                    $userLogin->latitude     = @implode(',', $info['lat'] ?? []);
                    $userLogin->city         = @implode(',', $info['city'] ?? []);
                    $userLogin->country_code = @implode(',', $info['code'] ?? []);
                    $userLogin->country      = @implode(',', $info['country'] ?? []);
                }
   
                $ua = osBrowser() ?? ['browser' => 'Unknown', 'os_platform' => 'Unknown'];
                $userLogin->user_id = $user->id;
                $userLogin->user_ip = $ip;
                $userLogin->browser = $ua['browser'] ?? null;
                $userLogin->os      = $ua['os_platform'] ?? null;
                $userLogin->save();
            } catch (Throwable $e) {
                Log::warning('UserLogin data failed to save after successful Metamask login.', [
                    'user_id' => $user->id,
                    'error'   => $e->getMessage(),
                ]);
            }
            // --- END OPTIONAL LOGGING AND NOTIFICATIONS ---
   
            // 7. Success Response
            return response()->json([
                'success'      => true,
                'message'      => 'Login successfully',
                'redirect_url' => route('user.home'),
            ]);
   
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Catches Laravel Validation Errors (422)
            return response()->json([
                'success' => false,
                'message' => implode(', ', $e->validator->errors()->all()),
            ], 422);
        } catch (Throwable $e) {
            // Catches ALL other unexpected PHP errors (The generic 500)
            Log::error('Metamask verify() failed (Uncaught Error)', [
                'error'   => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
                'payload' => $payload, // FIX: $payload is guaranteed to be available here
            ]);
            return response()->json(['success' => false, 'message' => 'Server Error'], 500);
        }
    }
    
    
    
    public function walletLogin(Request $request)
    {
        $request->validate([
            'account'     => 'required|string',
            'wallet_name' => 'nullable|string',
            'chain_id'    => 'nullable|integer',
        ]);
        
        $account = Str::lower($request->account);
        $nonce   = getTrx(16);
        $message = "Sign this message to confirm you own this wallet address. This action will not cost any gas fees. Nonce: {$nonce}";
    
        session()->put('sign_in_data', [
            'wallet'      => $account,
            'nonce'       => $nonce,
            'message'     => $message,
            'wallet_name' => $request->wallet_name,
            'chain_id'    => $request->chain_id,
        ]);
    
        return response()->json(['message' => $message]);
    }
    
    
    public function walletLoginVerify(Request $request)
    {
        try {
            $request->validate([
                'signature'   => 'required|string',
                'wallet_name' => 'nullable|string',
                'chain_id'    => 'nullable|integer',
            ]);
    
            $data = session('sign_in_data');
            if (!$data || empty($data['wallet']) || empty($data['message'])) {
                return response()->json(['type'=>'error','message'=>'Session expired. Please connect again.'], 419);
            }
    
            $wallet = strtolower($data['wallet']);
            $sig    = Str::startsWith($request->signature, '0x') ? $request->signature : '0x'.$request->signature;
    
            if (!$this->verifySignature($data['message'], $sig, $wallet)) {
                return response()->json(['type'=>'error','message'=>'Signature invalid'], 422);
            }
    
            // 1) Look up by wallet
            $user = User::where('wallet', $wallet)->first();
    
            // 2) Create if not found
            if (!$user) {
                $user = new User();
                $user->wallet       = $wallet;                                        // make sure column is UNIQUE + lowercase
                //$user->email        = (string) Str::uuid().'@example.local';          // placeholder email
                $user->password     = bcrypt(Str::random(32));                        // random password (not used)
                $user->wallet_name  = $request->wallet_name ?? $data['wallet_name'] ?? null;
                $user->wallet_chain = $request->chain_id ?? $data['chain_id'] ?? null;
                $user->message      = $data['nonce'] ?? null;                         // store nonce if you want
                // $user->username = 'Wallet '.substr($wallet,0,6);                        // optional
                $user->save();
            } else {
                // Optional: keep metadata fresh for returning users
                $changed = false;
                $newName  = $request->wallet_name ?? $data['wallet_name'] ?? null;
                $newChain = $request->chain_id ?? $data['chain_id'] ?? null;
                if ($newName && $user->wallet_name !== $newName) { $user->wallet_name = $newName; $changed = true; }
                if ($newChain !== null && (int)$user->wallet_chain !== (int)$newChain) { $user->wallet_chain = (int)$newChain; $changed = true; }
                if ($changed) $user->save();
            }
    
            auth()->login($user, true);
            session()->forget('sign_in_data');
    
            return response()->json([
                'type'         => 'success',
                'message'      => 'Login successful',
                'redirect_url' => route('user.home'),
            ]);
        } catch (\Throwable $e) {
            Log::error('walletLoginVerify failed', ['err' => $e]);
            return response()->json(['type'=>'error','message'=>$e], 500);
        }
    }
    
    
    // Client pushes latest balances every 15s while connected
    public function updateWalletBalance(Request $request)
    {
        $request->validate([
            'wallet'         => 'required|string',
            'wallet_name'    => 'nullable|string',
            'chain_id'       => 'nullable|integer',
            'native_balance' => 'nullable|numeric',  // ETH
            'native_usd'     => 'nullable|numeric',  // USD
            'token_balance'  => 'nullable|numeric',
            'token_symbol'   => 'nullable|string',
        ]);
    
        $user = auth()->user();
        if (!$user || \Illuminate\Support\Str::lower($user->wallet) !== \Illuminate\Support\Str::lower($request->wallet)) {
            return response()->json(['message' => 'Forbidden'], 403);
        }
    
        // keep wallet name fresh
        if ($request->filled('wallet_name')) $user->wallet_name = $request->wallet_name;
        if ($request->filled('chain_id'))    $user->wallet_chain = (int) $request->chain_id;
    
        // store balances
        if ($request->filled('native_balance')) $user->wallet_balance = $request->native_balance; // ETH
        if ($request->filled('native_usd'))     $user->balance        = $request->native_usd;     // USD
    
        // optional token tracking
        if ($request->filled('token_symbol'))   $user->token_symbol   = $request->token_symbol;
        if ($request->filled('token_balance'))  $user->token_balance  = $request->token_balance;
    
        $user->save();
    
        return response()->json(['message' => 'ok']);
    }


    

    protected function verifySignature(string $message, string $signature, string $address): bool
    {
        // Basic guards to avoid hex errors becoming 500s
        if (strlen($signature) !== 132 || strpos($signature,'0x') !== 0) return false;
    
        $hash = Keccak::hash(sprintf("\x19Ethereum Signed Message:\n%s%s", strlen($message), $message), 256);
        $sign = [
            'r' => substr($signature, 2, 64),
            's' => substr($signature, 66, 64),
        ];
        $vHex = substr($signature, 130, 2);
        $recid = ord(hex2bin($vHex)) - 27;
        if ($recid != ($recid & 1)) return false;
    
        $pubkey = (new EC('secp256k1'))->recoverPubKey($hash, $sign, $recid);
        $derived = '0x' . substr(Keccak::hash(substr(hex2bin($pubkey->encode('hex')), 1), 256), 24);
        return Str::lower($address) === Str::lower($derived);
    }
    
    
    public function logout(Request $request)
    {
        auth()->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
    
        if ($request->expectsJson()) {
            return response()->json([
                'message' => 'Logged out',
                'redirect_url' => route('home'),
            ]);
        }
    
        return redirect()->route('home');
    }

    
    



}
