<?php

namespace App\Http\Controllers\User\Auth\Web3;


use App\Http\Controllers\Controller;
use App\Models\AdminNotification;
use App\Models\User;
use App\Models\UserLogin;
use kornrunner\Keccak;
use Elliptic\EC;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;


class MetamaskController extends Controller
{
    public function message(Request $request)
    {
        $request->validate(['wallet_address' => 'required|string']);
    
        $address = Str::lower($request->wallet_address);
        $nonce   = strtoupper(getTrx());
    
        $message = gs('site_name') . " wants you to sign in with your Polygon account {$address}."
                 . " By sign in i'am agree with " . gs('site_name') . " privacy & policy."
                 . "\n\nNonce: {$nonce}\nIssued At: " . now();
    
        // store nonce for this address for 10 minutes
        Cache::put("siwe:{$address}", $nonce, now()->addMinutes(10));
    
        return response()->json([
            'success' => true,
            'address' => $address,
            'nonce'   => $nonce,
            'message' => $message,
        ]);
    }

    public function verify(Request $request)
    {
        try {
            // 1) Validate input
            $request->validate([
                'wallet_address' => ['required','string'],
                'message'        => ['required','string'],
                // 0x + 130 hex chars (r + s + v)
                'signature'      => ['required','string','regex:/^0x[0-9a-fA-F]{130}$/'],
                'chain_id'       => ['nullable','string'],
            ]);
    
            $address   = Str::lower($request->input('wallet_address'));
            $message   = $request->input('message');
            $signature = $request->input('signature');
    
            // 2) Extract nonce the user actually signed
            if (!preg_match('/Nonce:\s*([A-Z0-9]+)/', $message, $m)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Nonce not found in message.'
                ], 422);
            }
            $sentNonce = $m[1];
    
            // 3) Compare with the nonce we issued for this address
            $cachedNonce = Cache::pull("siwe:{$address}"); // pull = read & delete
            if (!$cachedNonce || !hash_equals($cachedNonce, $sentNonce)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid or expired nonce.'
                ], 422);
            }
    
            // 4) Cryptographically verify signature => address
            if (!$this->verifySignature($message, $signature, $address)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Signature verification failed.'
                ], 422);
            }
    
            // 5) Find or create the user
            $user = User::where('wallet', $address)->first();
            if (!$user) {
                    $user = new User();
                $user->wallet       = $address;                                        // make sure column is UNIQUE + lowercase
                //$user->email        = (string) Str::uuid().'@example.local';          // placeholder email
                $user->password     = bcrypt(Str::random(32));                        // random password (not used)
                $user->wallet_name  = 'MetaMask';
                $user->wallet_chain = $request->chain_id ?? null;
                $user->message      =  $sentNonce ?? null;                         // store nonce if you want
                // $user->username = 'Wallet '.substr($wallet,0,6);                        // optional
                $user->save();
    
                $adminNotification            = new AdminNotification();
                $adminNotification->user_id   = $user->id;
                $adminNotification->title     = 'New member registered';
                $adminNotification->click_url = urlPath('admin.users.detail', $user->id);
                $adminNotification->save();
            }
    
            // 6) Login and record footprint
            Auth::login($user, true);
    
            $ip        = getRealIP();
            $exist     = UserLogin::where('user_ip', $ip)->first();
            $userLogin = new UserLogin();
    
            if ($exist) {
                $userLogin->longitude    = $exist->longitude;
                $userLogin->latitude     = $exist->latitude;
                $userLogin->city         = $exist->city;
                $userLogin->country_code = $exist->country_code;
                $userLogin->country      = $exist->country;
            } else {
                $info                    = json_decode(json_encode(getIpInfo()), true);
                $userLogin->longitude    = @implode(',', $info['long']);
                $userLogin->latitude     = @implode(',', $info['lat']);
                $userLogin->city         = @implode(',', $info['city']);
                $userLogin->country_code = @implode(',', $info['code']);
                $userLogin->country      = @implode(',', $info['country']);
            }
    
            $ua = osBrowser();
            $userLogin->user_id = $user->id;
            $userLogin->user_ip = $ip;
            $userLogin->browser = $ua['browser'] ?? null;
            $userLogin->os      = $ua['os_platform'] ?? null;
            $userLogin->save();
    
            // 7) Success
            return response()->json([
                'success'      => true,
                'message'      => 'Login successfully',
                'redirect_url' => route('user.home'),
            ]);
    
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => implode(', ', $e->validator->errors()->all()),
            ], 422);
        } catch (\Throwable $e) {
            // Log full error so you can see exact cause in storage/logs/laravel.log
            Log::error('Metamask verify() failed', [
                'error'   => $e->getMessage(),
                'trace'   => $e->getTraceAsString(),
                'payload' => $request->all(),
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Server Error',
            ], 500);
        }
    }



    protected function verifySignature(string $message, string $signature, string $address): bool
    {
        $hash = Keccak::hash(sprintf("\x19Ethereum Signed Message:\n%s%s", strlen($message), $message), 256);
        $sign = [
            'r' => substr($signature, 2, 64),
            's' => substr($signature, 66, 64),
        ];
        $recid = ord(hex2bin(substr($signature, 130, 2))) - 27;

        if ($recid != ($recid & 1)) {
            return false;
        }

        $pubkey          = (new EC('secp256k1'))->recoverPubKey($hash, $sign, $recid);
        $derived_address = '0x' . substr(Keccak::hash(substr(hex2bin($pubkey->encode('hex')), 1), 256), 24);
        return (Str::lower($address) === $derived_address);
    }
}
