<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller
{
    public function profile()
    {
        $pageTitle = "Profile Settings";
        $user      = auth()->user()->load('userRanking');
    
    
        $countries        = json_decode(file_get_contents(resource_path('views/partials/country.json')));
        return view($this->activeTemplate . 'user.profile_setting', compact('pageTitle', 'user', 'countries'));
    }

    public function submitProfile(Request $request)
    {
        $user = auth()->user();
    
        /* -------- Countries map (dial + name) -------- */
        $countryData = json_decode(file_get_contents(resource_path('views/partials/country.json')), true);
        $countryKeys = array_keys((array) $countryData);
        $countriesCsv = implode(',', $countryKeys);
    
        // Prefer hidden fields populated by JS; fall back to select value
        $countryCode = $request->input('country_code', $request->input('country')); // e.g. "BD"
        $mobileCode  = $request->input('mobile_code');                               // e.g. "880"
    
        $countryRow  = ($countryCode && isset($countryData[$countryCode])) ? $countryData[$countryCode] : null;
        $countryName = $countryRow['country']   ?? null;
        $dialCode    = $mobileCode ?: ($countryRow['dial_code'] ?? '');
    
        /* ---------------- Validation ----------------- */
        $request->validate([
            'firstname'   => 'required|string',
            'lastname'    => 'required|string',
            'email'       => 'required|email|string|max:191|unique:users,email,' . $user->id,
            'mobile'      => 'nullable|string|max:40|unique:users,mobile,' . $user->id,
            'country'     => 'required|in:' . $countriesCsv, // select name="country"
            'profile_pic' => 'nullable|image|mimes:jpg,jpeg,png,webp', // 5 MB
            'cover_pic'   => 'nullable|image|mimes:jpg,jpeg,png,webp', // 8 MB
            'remove_profile_pic' => 'nullable|in:0,1',
            'remove_cover_pic'   => 'nullable|in:0,1',
        ], [
            'firstname.required' => 'First name field is required',
            'lastname.required'  => 'Last name field is required',
        ]);
    
        /* --------- Normalize & set the mobile -------- */
        // Keep only digits from the visible input
        $cleanMobile = preg_replace('/\D+/', '', (string) $request->mobile);
    
        // Digits only for the dial code
        $dialDigits  = preg_replace('/\D+/', '', (string) $dialCode);
    
        // If the visible input already begins with the dial, strip it once (prevents +880880…)
        if ($dialDigits && str_starts_with($cleanMobile, $dialDigits)) {
            $cleanMobile = substr($cleanMobile, strlen($dialDigits));
        }
    
        // Final storage: +<dial><local>
        $finalMobile = ($dialDigits ? '+' . $dialDigits : '') . $cleanMobile;
    
        /* --------------- Assign fields --------------- */
        $user->firstname    = $request->firstname;
        $user->lastname     = $request->lastname;
        $user->email        = $request->email;
        $user->mobile       = $finalMobile;
        $user->country_code = $countryCode;
    
        // Address object – overwrite with new values; keep existing if not provided
        $user->address = [
            'address'       => $request->input('address', $user->address->address ?? null),
            'state'         => $request->input('state',   $user->address->state   ?? null),
            'zip'           => $request->input('zip',     $user->address->zip     ?? null),
            'city'          => $request->input('city',    $user->address->city    ?? null),
            'country'       => $countryName ?: ($user->address->country ?? null),
            'country_code'  => $countryCode,
            'mobile_code'   => $dialDigits,
        ];
    
        /* ---- Handle “Remove” (delete old without new upload) ---- */
        // Helper to delete a stored file safely (adjust if you have a project-wide helper)
        $deleteStoredFile = function (string $basePath, ?string $filename): void {
            if (!$filename) return;
            $full = rtrim($basePath, '/').'/'.$filename;
            if (file_exists($full)) { @unlink($full); }
            // If you also store thumbs, delete them here as well.
        };
    
        if ($request->boolean('remove_profile_pic')) {
            // Delete file from disk and clear DB field
            $deleteStoredFile(getFilePath('userProfile'), $user->profile_pic);
            $user->profile_pic = null;
        }
    
        if ($request->boolean('remove_cover_pic')) {
            $deleteStoredFile(getFilePath('userCover'), $user->cover_pic);
            $user->cover_pic = null;
        }
    
        /* ---- Uploads (replace old using your fileUploader) ---- */
        if ($request->hasFile('profile_pic')) {
            // Pass the OLD filename so FileManager removes it
            $user->profile_pic = fileUploader(
                $request->file('profile_pic'),
                getFilePath('userProfile'),
                getFileSize('userProfile'),
                $user->profile_pic // old
                // , getFileThumb('userProfile') // add if you use thumbs
            );
        }
    
        if ($request->hasFile('cover_pic')) {
            $user->cover_pic = fileUploader(
                $request->file('cover_pic'),
                getFilePath('userCover'),
                getFileSize('userCover'),
                $user->cover_pic // old
                // , getFileThumb('userCover')
            );
        }
    
        $user->save();
    
        $notify[] = ['success', 'Profile Updated Successfully'];
        return back()->withNotify($notify);
    }



    public function changePassword()
    {
        $pageTitle = 'Change Password';
        return view($this->activeTemplate . 'user.password', compact('pageTitle'));
    }

    public function submitPassword(Request $request)
    {
        $passwordValidation = Password::min(6);
        if (gs('secure_password')) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

        $this->validate($request, [
            'current_password' => 'required',
            'password'         => ['required', 'confirmed', $passwordValidation],
        ]);

        $user = auth()->user();
        if (Hash::check($request->current_password, $user->password)) {
            $password       = Hash::make($request->password);
            $user->password = $password;
            $user->save();
            $notify[] = ['success', 'Password Changed Successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'The password doesn\'t match!'];
            return back()->withNotify($notify);
        }
    }
}
