<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Lib\FormProcessor;
use App\Lib\GoogleAuthenticator;
use App\Lib\HyipLab;
use App\Models\Deposit;
use App\Models\Form;
use App\Models\Invest;
use App\Models\PromotionTool;
use App\Models\Frontend;
use App\Models\Referral;
use App\Models\SupportTicket;
use App\Models\Transaction;
use App\Models\User;
use App\Models\GeneralSetting;
use App\Models\Withdrawal;
use App\Models\UserRanking;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;


class UserController extends Controller
{
    public function home()
    {
        $data['pageTitle']         = 'Dashboard';
        $user                      = auth()->user();
        $data['user']              = $user;
        

        $data['chartData'] = Transaction::where('remark', 'interest')
            ->where('created_at', '>=', Carbon::now()->subDays(30))
            ->where('user_id', $user->id)
            ->selectRaw("SUM(amount) as amount, DATE_FORMAT(created_at,'%Y-%m-%d') as date")
            ->orderBy('created_at', 'asc')
            ->groupBy('date')
            ->get();

        return view($this->activeTemplate . 'user.dashboard', $data);
    }
    
    public function cryptoMarket()
    {
        $pageTitle = 'Live Crypto Market';
     
        return view($this->activeTemplate . 'user.crypto_market', compact('pageTitle'));
    }
    
    public function storeMgrPurchase(Request $request)
    {
        $user = auth()->user();
    
        // Validate input and log validation issues
        try {
            $data = $request->validate([
                'tx_hash'      => 'required|string|max:100',
                'from_address' => 'required|string|max:100',
                'pol_amount'   => 'required|numeric|min:0',
                'mgr_amount'   => 'required|numeric|min:0',
                'to_address'   => 'nullable|string|max:100',
            ]);
        } catch (\Throwable $e) {
            Log::warning('MGR purchase validation failed', [
                'user_id' => optional($user)->id,   // <- no ?-> here
                'payload' => $request->all(),
                'error'   => $e->getMessage(),
            ]);
            return response()->json(['ok' => false, 'message' => 'Invalid payload'], 422);
        }
    
        // Idempotency
        if (Transaction::where('trx', $data['tx_hash'])->where('remark', 'mgr_purchase')->exists()) {
            return response()->json(['ok' => true, 'duplicate' => true]);
        }
    
        try {
            DB::transaction(function () use ($user, $data) {
                $mgrAmount = (float)$data['mgr_amount'];
                $polAmount = (float)$data['pol_amount'];
    
                // 1) Update user's MGR balance and totals
                $user->token_balance = (float)($user->token_balance ?? 0) + $mgrAmount;
                if (Schema::hasColumn('users', 'total_invests')) {
                    $user->total_invests = (float)($user->total_invests ?? 0) + $mgrAmount;
                }
                $user->save();
    
                // 2) Ledger transaction (+)
                $t = new Transaction();
                $t->user_id      = $user->id;
                $t->amount       = $mgrAmount;
                $t->post_balance = $user->token_balance;
                $t->charge       = 0;
                $t->trx_type     = '+';
                $t->details      = 'MGR purchase via presale';
                $t->trx          = $data['tx_hash'];
                $t->wallet_type  = 'MGR';
                $t->remark       = 'mgr_purchase';
                $t->save();
    
                // 3) Invests entry (as purchase log)
                $inv = new Invest();
                $inv->user_id            = $user->id;
                $inv->plan_id            = null;
                $inv->amount             = $mgrAmount;
                $inv->initial_amount     = $mgrAmount;
                $inv->interest           = 0;
                $inv->initial_interest   = 0;
                $inv->period             = 0;
                $inv->time_name          = 'instant';
                $inv->hours              = 0;
                $inv->next_time          = now();
                $inv->should_pay         = 0;
                $inv->status             = 1;
                $inv->wallet_type        = 'MGR';
                $inv->capital_status     = 0;
                $inv->trx                = $data['tx_hash'];
                $inv->compound_times     = 0;
                $inv->rem_compound_times = 0;
                $inv->hold_capital       = 0;
    
                if (Schema::hasColumn('invests', 'pol_amount'))   $inv->pol_amount   = $polAmount;
                if (Schema::hasColumn('invests', 'from_address')) $inv->from_address = $data['from_address'];
                if (Schema::hasColumn('invests', 'to_address'))   $inv->to_address   = $data['to_address'] ?? null;
    
                $inv->save();
    
                // 4) Level commissions
                try {
                    $general = function_exists('gs') ? gs() : (config('app.general') ?? null);
                    \App\Lib\HyipLab::levelCommission(
                        $user,
                        $mgrAmount,
                        'deposit_commission',
                        $data['tx_hash'],
                        $general
                    );
                } catch (\Throwable $e) {
                    Log::error('MGR commission error', [
                        'user_id' => optional($user)->id,
                        'tx_hash' => $data['tx_hash'],
                        'error'   => $e->getMessage(),
                        'file'    => $e->getFile(),
                        'line'    => $e->getLine(),
                    ]);
                    // keep purchase persisted even if commission fails
                }
            });
    
            return response()->json(['ok' => true]);
    
        } catch (\Throwable $e) {
            Log::error('MGR purchase store error', [
                'user_id' => optional($user)->id,
                'tx_hash' => $data['tx_hash'] ?? null,
                'payload' => $data ?? [],
                'error'   => $e->getMessage(),
                'file'    => $e->getFile(),
                'line'    => $e->getLine(),
                'trace'   => $e->getTraceAsString(),
            ]);
            return response()->json(['ok' => false, 'message' => 'Server error'], 500);
        }
    }
    
    
    public function getMgrMarket()
    {
        $g  = GeneralSetting::first();
        
        return response()->json([
            'market_cap' => (string) $g->market_cap, // keep as strings
            'liquidity'  => (string) $g->liquidity,
            'mgr_amount'  => (string) $g->mgr_amount,
            'fake_trns'  => (string) $g->fake_trns,
            'min_trade'  => (string) $g->min_trade,
            'max_trade'  => (string) $g->max_trade,
        ]);
    }
    
    public function recentMgrPurchases()
    {
        // Use your Invest or Transaction source. Example with Invest:
        $since = now()->subSeconds(60);
        $rows  = \App\Models\Invest::where('user_id', '!=', null) // or remove filter if global feed
                   ->where('status', 1)
                   ->where('created_at', '>=', $since)
                   ->orderByDesc('id')
                   ->limit(50)
                   ->get(['trx as tx_hash', 'created_at', 'amount as mgr_amount', 'pol_amount']);
    
        // Normalize for frontend
        return response()->json($rows->map(function($r){
            return [
                'tx_hash'    => (string)($r->tx_hash ?? ''),
                'created_at' => $r->created_at?->toIso8601String(),
                'mgr_amount' => (string)$r->mgr_amount,
                'pol_amount' => $r->pol_amount !== null ? (string)$r->pol_amount : null,
            ];
        }));
    }



    
    
   public function teamMember()
    {
        $pageTitle = 'Team Member';
        $auth      = auth()->user();
        
          
        $maxLevel  = 8;
    
        $levels = [];
        $currentIds = collect([$auth->id]);
    
        for ($level = 1; $level <= $maxLevel; $level++) {
            if ($currentIds->isEmpty()) break;
    
            $members = \App\Models\User::query()
                ->whereIn('ref_by', $currentIds)
                ->withCount('referrals')
                ->withSum('invests', 'amount')
                ->with(['referrer:id,firstname,lastname,username']) // 👈 eager-load referrer
                ->orderBy('id', 'asc')
                ->get(['id','firstname','lastname','username','email','profile_pic','created_at','ref_by']);
    
            if ($members->isEmpty()) break;
    
            $levels[$level] = $members;
            $currentIds = $members->pluck('id');
        }
    
        $general = gs() ?? (object)['cur_sym' => '$'];
    
        return view($this->activeTemplate . 'user.team_member', compact(
            'pageTitle','auth','levels','maxLevel','general'
        ));
    }

    public function depositHistory(Request $request)
    {
        $pageTitle = 'Deposit History';
        $deposits  = auth()->user()->deposits()->searchable(['trx'])->with(['gateway'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view($this->activeTemplate . 'user.deposit_history', compact('pageTitle', 'deposits'));
    }
    
    
    public function teamCommission()
    {
        $pageTitle = 'Team Commission';
        $user      = auth()->user();
    
        // Levels and their commission percentages
        $levels = [
            1 => 15,
            2 => 4,
            3 => 3,
            4 => 3,
            5 => 3,
            6 => 3,
            7 => 3,
            8 => 2,
        ];
    
        // Unlock requirements (token balance needed to unlock sets of levels)
        $requirements = [
            '1-2' => 20000,
            '3-5' => 60000,
            '6-8' => 120000,
        ];
    
        $tokenBalance = (float) ($user->token_balance ?? 0);
    
        return view($this->activeTemplate . 'user.team_commission',
            compact('pageTitle','levels','requirements','tokenBalance','user'));
    }
    
    
    public function teamReward()
    {
        $pageTitle = 'Team Reward';
        $user      = auth()->user();
    
        // ---- Build team investment by *exact* level depth (1..8) ----
        $maxDepth = 8;
        $levelSums = [];
        $levelCounts = [];
        $currentIds = collect([$user->id]);
    
        for ($level = 1; $level <= $maxDepth; $level++) {
            if ($currentIds->isEmpty()) {
                $levelSums[$level]   = 0.0;
                $levelCounts[$level] = 0;
                continue;
            }
    
            $members = \App\Models\User::query()
                ->whereIn('ref_by', $currentIds)
                ->get(['id', 'token_balance']);
    
            $levelSums[$level]   = (float) $members->sum('token_balance');
            $levelCounts[$level] = (int)   $members->count();
    
            $currentIds = $members->pluck('id'); // next layer
        }
    
        $teamInvestTotal = array_sum($levelSums); // total MGR in team
    
        // ---- Fetch ranks and compute progress for each ----
        $ranks = UserRanking::query()
            ->when(method_exists(UserRanking::class, 'scopeActive'), fn($q) => $q->active())
            ->orderBy('team_invest','asc')
            ->get();
    
        $rankCards = $ranks->map(function ($r) use ($user, $teamInvestTotal) {
            $required = (float) $r->team_invest;
            $have     = (float) $teamInvestTotal;
            $pct      = $required > 0 ? min(100, round(($have / $required) * 100)) : 100;
            $done     = $user->user_ranking_id && $r->id <= (int) $user->user_ranking_id;
    
            return [
                'id'          => $r->id,
                'level'       => (int) $r->level,
                'name'        => $r->name,
                'required'    => $required,           // team invest needed (MGR)
                'current'     => $have,               // current total (MGR)
                'progress'    => $pct,
                'completed'   => $done,
                'bonus_token' => (float) $r->bonus_token, // MGR
                'bonus_usd'   => (float) $r->bonus_usd,   // $
            ];
        })->values();
    
        return view($this->activeTemplate . 'user.team_reward', compact(
            'pageTitle',
            'user',
            'levelSums',
            'levelCounts',
            'teamInvestTotal',
            'rankCards'
        ));
    }


    public function show2faForm()
    {
        $ga        = new GoogleAuthenticator();
        $user      = auth()->user();
        $secret    = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($user->username . '@' . gs('site_name'), $secret);
        $pageTitle = '2FA Setting';
        return view($this->activeTemplate . 'user.twofactor', compact('pageTitle', 'secret', 'qrCodeUrl'));
    }

    public function create2fa(Request $request)
    {
        $user = auth()->user();
        $this->validate($request, [
            'key'  => 'required',
            'code' => 'required',
        ]);
        $response = verifyG2fa($user, $request->code, $request->key);
        if ($response) {
            $user->tsc = $request->key;
            $user->ts  = 1;
            $user->save();
            $notify[] = ['success', 'Google authenticator activated successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'Wrong verification code'];
            return back()->withNotify($notify);
        }
    }

    public function disable2fa(Request $request)
    {
        $this->validate($request, [
            'code' => 'required',
        ]);

        $user     = auth()->user();
        $response = verifyG2fa($user, $request->code);
        if ($response) {
            $user->tsc = null;
            $user->ts  = 0;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator deactivated successfully'];
        } else {
            $notify[] = ['error', 'Wrong verification code'];
        }
        return back()->withNotify($notify);
    }

    public function transactions()
    {
        $pageTitle = 'Transactions';
        $remarks   = Transaction::distinct('remark')->orderBy('remark')->get('remark');

        $transactions = Transaction::where('user_id', auth()->id())->searchable(['trx'])->filter(['trx_type', 'remark', 'wallet_type'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view($this->activeTemplate . 'user.transactions', compact('pageTitle', 'transactions', 'remarks'));
    }

    public function kycForm()
    {
        if (auth()->user()->kv == 2) {
            $notify[] = ['error', 'Your KYC is under review'];
            return to_route('user.home')->withNotify($notify);
        }
        if (auth()->user()->kv == 1) {
            $notify[] = ['error', 'You are already KYC verified'];
            return to_route('user.home')->withNotify($notify);
        }
        $pageTitle = 'KYC Form';
        $form      = Form::where('act', 'kyc')->first();
        return view($this->activeTemplate . 'user.kyc.form', compact('pageTitle', 'form'));
    }

    public function kycData()
    {
        $user      = auth()->user();
        $pageTitle = 'KYC Data';
        return view($this->activeTemplate . 'user.kyc.info', compact('pageTitle', 'user'));
    }

    public function kycSubmit(Request $request)
    {
        $form           = Form::where('act', 'kyc')->first();
        $formData       = $form->form_data;
        $formProcessor  = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);

        $userData       = $formProcessor->processFormData($request, $formData);
        $user           = auth()->user();
        $user->kyc_data = $userData;
        $user->kv       = 2;
        $user->save();

        $notify[] = ['success', 'KYC data submitted successfully'];
        return to_route('user.home')->withNotify($notify);

    }

    public function attachmentDownload($fileHash)
    {
        $filePath  = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $general   = gs();
        $title     = slug($general->site_name) . '- attachments.' . $extension;
        $mimetype  = mime_content_type($filePath);
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }

    public function userData()
    {
        $user = auth()->user();
        if ($user->profile_complete == 1) {
            return to_route('user.home');
        }
        $pageTitle = "Personal Information";
        return view($this->activeTemplate . 'user.user_data', compact('pageTitle', 'user'));
    }

    public function userDataSubmit(Request $request)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();
    
        if ($user->profile_complete == 1) {
            return to_route('user.home');
        }
    
        // Request first, then session; treat empty strings as "not provided"
        $reference = $request->filled('reference')
            ? trim($request->input('reference'))
            : trim((string) session('reference', ''));
    
        // Track whether we actually used the session value so we can clear it later
        $usedSessionReference = !$request->filled('reference') && $reference !== '';
    
        // Validate core fields
        $validated = $request->validate([
            'firstname' => ['required', 'string', 'max:255'],
            'lastname'  => ['required', 'string', 'max:255'],
            'phone'     => ['nullable', 'string', 'max:50'],
            'email'     => [
                'nullable',
                'email',
                'max:255',
                Rule::unique('users', 'email')->ignore($user->id),
            ],
            'reference' => ['nullable', 'string', 'max:255'],
        ]);
    
        // Resolve referral user if reference present (from request or session)
        $referUser = null;
        if ($reference !== '') {
            $referUser = User::where('username', $reference)->first();
    
            if (!$referUser) {
                $notify[] = ['error', 'Invalid Referral Code'];
                return back()->withNotify($notify)->withInput();
            }
    
            // Prevent self-referrals
            if ((int) $referUser->id === (int) $user->id) {
                $notify[] = ['error', 'You cannot use your own referral code'];
                return back()->withNotify($notify)->withInput();
            }
        }
    
        // Update only allowed fields
        $user->firstname = $validated['firstname'];
        $user->lastname  = $validated['lastname'];
        $user->mobile    = $validated['phone'] ?? $user->mobile;
        $user->email     = $validated['email'] ?? $user->email;
    
        $user->ref_by = $referUser ? $referUser->id : null;
    
        $user->profile_complete = 1;
        $user->save();
    
        // Clear session referral ONLY if we actually used it
        if ($usedSessionReference) {
            session()->forget('reference');
        }
    
        $notify[] = ['success', 'Registration process completed successfully'];
        return to_route('user.home')->withNotify($notify);
    }
    



    public function referrals()
    {
        $pageTitle = 'Referrals';
    
        $auth     = auth()->user();
        $authId   = $auth->id;
        $fallback = Referral::max('level') ?: 1;
    
        $maxLevel = method_exists($this, 'getMaxReferralLevel')
            ? ($this->getMaxReferralLevel($auth) ?: $fallback)
            : $fallback;
    
        /** ---------- Eager-load full subtree (no N+1) ---------- */
        // Load invests & referrer on every node + referrals at each depth.
        $with = ['invests', 'referrer'];
        $path = 'referrals';
        for ($i = 1; $i <= $maxLevel; $i++) {
            $with[] = $path;
            $with[] = $path . '.invests';
            $with[] = $path . '.referrer';
            $path  .= '.referrals';
        }
    
        /** @var \App\Models\User $user */
        $user = \App\Models\User::with($with)->findOrFail($authId);
    
        /** ---------- Level-wise totals (only loaded relations) ---------- */
        $totalTeamMembers    = 0;
        $totalTeamInvestment = 0.0;
        $levelData           = [];  // only non-empty levels go here
    
        $currentLevelUsers = collect([$user]);
    
        for ($level = 1; $level <= $maxLevel; $level++) {
            $teamMembersCount = 0;
            $teamInvestment   = 0.0;
            $nextLevelUsers   = collect();
    
            foreach ($currentLevelUsers as $parent) {
                $refs  = $parent->referrals; // already eager-loaded
                $count = $refs->count();
    
                if ($count > 0) {
                    $teamMembersCount += $count;
                    $teamInvestment   += $refs->sum(fn ($r) => $r->invests->sum('amount'));
                    $nextLevelUsers    = $nextLevelUsers->merge($refs);
                }
            }
    
            if ($teamMembersCount > 0) {
                $levelData[$level] = [
                    'team_members' => $teamMembersCount,
                    'investment'   => $teamInvestment,
                ];
                $totalTeamMembers    += $teamMembersCount;
                $totalTeamInvestment += $teamInvestment;
            }
    
            if ($nextLevelUsers->isEmpty()) {
                break; // stop early if no deeper users
            }
            $currentLevelUsers = $nextLevelUsers;
        }
    
        // No Treant data — the blade builds the tree directly from $user
        return view($this->activeTemplate . 'user.referrals', compact(
            'pageTitle',
            'user',
            'maxLevel',
            'totalTeamMembers',
            'totalTeamInvestment',
            'levelData'
        ));
    }
    
    public function newsUpdate()
    {
        $pageTitle = 'News And Update';
        $news     = Frontend::where('data_keys', 'news.element')->where('template_name', activeTemplateName())->orderBy('id', 'desc')->paginate(getPaginate(9));

        return view($this->activeTemplate . 'user.news_update', compact('pageTitle', 'news'));
    }





    

    /**
     * Recursively calculate the max referral level
     */
    private function getMaxReferralLevel($user, $level = 1)
    {
        $directReferrals = $user->referrals;
    
        if ($directReferrals->isEmpty()) {
            return $level - 1; // No more referrals at this level
        }
    
        $maxSubLevel = 0;
        foreach ($directReferrals as $referral) {
            $subLevel = $this->getMaxReferralLevel($referral, $level + 1);
            $maxSubLevel = max($maxSubLevel, $subLevel);
        }
    
        return max($level, $maxSubLevel);
    }


 
    
    
  
    public function promotionalBanners()
    {
        $promotionCount = PromotionTool::count();
        if (!gs('promotional_tool') || !$promotionCount) {
            abort(404);
        }
        $pageTitle    = 'Promotional Banners';
        $banners      = PromotionTool::orderBy('id', 'desc')->get();
        $emptyMessage = 'No banner found';
        return view($this->activeTemplate . 'user.promo_tools', compact('pageTitle', 'banners', 'emptyMessage'));
    }

    public function transferBalance()
    {
        if (!gs('b_transfer')) {
            abort(404);
        }
        $pageTitle = 'Transfer - Others';
        $user      = auth()->user();
        return view($this->activeTemplate . 'user.balance_transfer', compact('pageTitle', 'user'));
    }

    public function transferBalanceSubmit(Request $request)
    {
        $general = gs();
        if (!$general->b_transfer) {
            abort(404);
        }
        $request->validate([
            'username' => 'required',
            'amount'   => 'required|numeric|gt:0',
            'wallet'   => 'required|in:deposit_wallet,return_walltet',
        ]);

        $user = auth()->user();
        if ($user->username == $request->username) {
            $notify[] = ['error', 'You cannot transfer Balance to your own Account'];
            return back()->withNotify($notify);
        }

        $receiver = User::where('username', $request->username)->first();
        if (!$receiver) {
            $notify[] = ['error', 'Invalid Recipient Details'];
            return back()->withNotify($notify);
        }

        if ($user->ts) {
            $response = verifyG2fa($user, $request->authenticator_code);
            if (!$response) {
                $notify[] = ['error', 'Wrong Verification Code'];
                return back()->withNotify($notify);
            }
        }

        $general     = gs();
        $charge      = $general->f_charge + ($request->amount * $general->p_charge) / 100;
        $afterCharge = $request->amount + $charge;
        $wallet      = $request->wallet;

        if ($user->$wallet < $afterCharge) {
            $notify[] = ['error', 'Insufficient Balance'];
            return back()->withNotify($notify);
        }

        $user->$wallet -= $afterCharge;
        $user->save();

        $trx1                      = getTrx();
        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = getAmount($afterCharge);
        $transaction->charge       = $charge;
        $transaction->trx_type     = '-';
        $transaction->trx          = $trx1;
        $transaction->wallet_type  = $wallet;
        $transaction->remark       = 'balance_transfer';
        $transaction->details      = 'Balance transfer to ' . $receiver->username;
        $transaction->post_balance = getAmount($user->$wallet);
        $transaction->save();

        $receiver->$wallet += $request->amount;
        $receiver->save();

        $trx2                      = getTrx();
        $transaction               = new Transaction();
        $transaction->user_id      = $receiver->id;
        $transaction->amount       = getAmount($request->amount);
        $transaction->charge       = 0;
        $transaction->trx_type     = '+';
        $transaction->trx          = $trx2;
        $transaction->wallet_type  = $wallet;
        $transaction->remark       = 'balance_received';
        $transaction->details      = 'Balance received from ' . $user->username;
        $transaction->post_balance = getAmount($user->$wallet);
        $transaction->save();

        notify($user, 'BALANCE_TRANSFER', [
            'amount'        => showAmount($request->amount),
            'charge'        => showAmount($charge),
            'wallet_type'   => keyToTitle($wallet),
            'post_balance'  => showAmount($user->$wallet),
            'user_fullname' => $receiver->fullname,
            'username'      => $receiver->username,
            'trx'           => $trx1,
        ]);

        notify($receiver, 'BALANCE_RECEIVE', [
            'wallet_type'  => keyToTitle($wallet),
            'amount'       => showAmount($request->amount),
            'post_balance' => showAmount($receiver->$wallet),
            'sender'       => $user->username,
            'trx'          => $trx2,
        ]);

        $notify[] = ['success', 'Balance Transferred Successfully'];
        return back()->withNotify($notify);
    }

    public function findUser(Request $request)
    {
        $user    = User::where('username', $request->username)->first();
        $message = null;
        if (!$user) {
            $message = 'User not found';
        }
        if (@$user->username == auth()->user()->username) {
            $message = 'You cannot transfer Balance to your own Account';
        }
        return response(['message' => $message]);
    }
    
    public function convertBalance()
    {
        
        $pageTitle = 'Transfer - Self';
        $user      = auth()->user();
        return view($this->activeTemplate . 'user.balance_convert', compact('pageTitle', 'user'));
    }
    
      public function convertBalanceSubmit(Request $request)
    {
        $general = gs();
      
      
        $request->validate([
            'amount'   => 'required|numeric|gt:0',
            'wallet'   => 'required|in:referral_walltet,interest_wallet',
        ]);

        $user = auth()->user();

        $general     = gs();
        //$charge      = $general->f_charge + ($request->amount * $general->p_charge) / 100;
        $charge      = 0;
        $afterCharge = $request->amount;
        $wallet      = $request->wallet;

        if ($user->$wallet < $afterCharge) {
            $notify[] = ['error', 'Insufficient Balance'];
            return back()->withNotify($notify);
        }

        $user->$wallet -= $afterCharge;
        $user->deposit_wallet += $afterCharge;
        $user->save();

        /*
        $trx1                      = getTrx();
        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = getAmount($afterCharge);
        $transaction->charge       = $charge;
        $transaction->trx_type     = '+';
        $transaction->trx          = $trx1;
        $transaction->wallet_type  = $wallet;
        $transaction->remark       = 'balance_convert';
        $transaction->details      = 'Balance Convert '.$wallet. ' to Reserve Wallet';
        $transaction->post_balance = getAmount($user->$wallet);
        $transaction->save();
        */


        notify($user, 'BALANCE_CONVERT', [
            'amount'        => showAmount($request->amount),
            'charge'        => showAmount($charge),
            'wallet_type'   => keyToTitle($wallet),
            'post_balance'  => showAmount($user->$wallet),
            'user_fullname' => $user->fullname,
            'username'      => $user->username,
            'trx'           => getTrx(),
        ]);


        $notify[] = ['success', 'Balance Transferred Successfully'];
        return back()->withNotify($notify);
    }

}
