<?php

namespace App\Lib;

use App\Models\AdminNotification;
use App\Models\Holiday;
use App\Models\Invest;
use App\Models\GeneralSetting;
use App\Models\Referral;
use App\Models\Transaction;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class HyipLab
{
    /**
     * Instance of investor user
     *
     * @var object
     */
    private $user;

    /**
     * Plan which is purchasing
     *
     * @var object
     */
    private $plan;

    /**
     * General setting
     *
     * @var object
     */
    private $setting;

    /**
     * Set some properties
     *
     * @param object $user
     * @param object $plan
     * @return void
     */
    public function __construct($user, $plan)
    {
        $this->user    = $user;
        $this->plan    = $plan;
        $this->setting = gs();
    }

    /**
     * Invest process
     *
     * @param float $amount
     * @param string $wallet
     * @return void
     */
    public function invest($amount, $wallet, $compoundTimes = 0)
    {
        $plan = $this->plan;
        $user = $this->user;

        $user->$wallet -= $amount;
        $user->total_invests += $amount;
        $user->save();

        $trx                       = getTrx();
        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $amount;
        $transaction->post_balance = $user->$wallet;
        $transaction->charge       = 0;
        $transaction->trx_type     = '-';
        $transaction->details      = 'Invested on ' . $plan->name;
        $transaction->trx          = $trx;
        $transaction->wallet_type  = $wallet;
        $transaction->remark       = 'invest';
        $transaction->save();

        //start
        if ($plan->interest_type == 1) {
            $interestAmount = ($amount * $plan->interest) / 100;
        } else {
            $interestAmount = $plan->interest;
        }

        $period = ($plan->lifetime == 1) ? -1 : $plan->repeat_time;

        $next = self::nextWorkingDay($plan->timeSetting->time);

        $shouldPay = -1;
        if ($period > 0) {
            $shouldPay = $interestAmount * $period;
        }

        $invest                     = new Invest();
        $invest->user_id            = $user->id;
        $invest->plan_id            = $plan->id;
        $invest->amount             = $amount;
        $invest->initial_amount     = $amount;
        $invest->interest           = $interestAmount;
        $invest->initial_interest   = $interestAmount;
        $invest->period             = $period;
        $invest->time_name          = $plan->timeSetting->name;
        $invest->hours              = $plan->timeSetting->time;
        $invest->next_time          = $next;
        $invest->should_pay         = $shouldPay;
        $invest->status             = 1;
        $invest->wallet_type        = $wallet;
        $invest->capital_status     = $plan->capital_back;
        $invest->trx                = $trx;
        $invest->compound_times     = $compoundTimes ?? 0;
        $invest->rem_compound_times = $compoundTimes ?? 0;
        $invest->hold_capital       = $plan->hold_capital;
        $invest->save();

        if ($this->setting->invest_commission == 1) {
            $commissionType = 'invest_commission';
            self::levelCommission($user, $amount, $commissionType, $trx, $this->setting);
        }

        notify($user, 'INVESTMENT', [
            'trx'             => $invest->trx,
            'amount'          => showAmount($amount),
            'plan_name'       => $plan->name,
            'interest_amount' => showAmount($interestAmount),
            'time'            => $plan->lifetime == 1 ? 'lifetime' : $plan->repeat_time . ' times',
            'time_name'       => $plan->timeSetting->name,
            'wallet_type'     => keyToTitle($wallet),
            'post_balance'    => showAmount($user->$wallet),
        ]);

        $adminNotification            = new AdminNotification();
        $adminNotification->user_id   = $user->id;
        $adminNotification->title     = $this->setting->cur_sym . showAmount($amount) . ' invested to ' . $plan->name;
        $adminNotification->click_url = '#';
        $adminNotification->save();

        while ($user->ref_by) {
            $user = User::find($user->ref_by);
            $user->team_invests += $amount;
            $user->save();
        }
    }

    /**
     * Get the next working day of the system
     *
     * @param integer $hours
     * @return string
     */
    public static function nextWorkingDay($hours)
    {
        $now     = now();
        $setting = gs();
        while (0 == 0) {
            $nextPossible = Carbon::parse($now)->addHours($hours)->toDateTimeString();

            if (!self::isHoliDay($nextPossible, $setting)) {
                $next = $nextPossible;
                break;
            }
            $now = $now->addDay();
        }
        return $next;
    }

    /**
     * Check the date is holiday or not
     *
     * @param string $date
     * @param object $setting
     * @return string
     */
    public static function isHoliDay($date, $setting)
    {
        $isHoliday = true;
        $dayName   = strtolower(date('D', strtotime($date)));
        $holiday   = Holiday::where('date', date('Y-m-d', strtotime($date)))->count();
        $offDay    = (array) $setting->off_day;

        if (!array_key_exists($dayName, $offDay)) {
            if ($holiday == 0) {
                $isHoliday = false;
            }
        }

        return $isHoliday;

    }
    
    
    public static function isHolidayRoi($date, $setting)
    {
        $isHolidayroi = true;
        $dayName   = strtolower(date('D', strtotime($date)));
        $offDay    = (array) $setting->off_day_roi;

        if (!array_key_exists($dayName, $offDay)) {
            $isHolidayroi = false;
        }

        return $isHolidayroi;

    }

    /**
     * Give referral commission
     *
     * @param object $user
     * @param float $amount
     * @param string $commissionType
     * @param string $trx
     * @param object $setting
     * @return void
     */
     
    public static function levelCommission($user, $amount, $commissionType, $trx, $setting)
    {
        // Commission tiers for this type, keyed by level (1..N)
        $tiers = \App\Models\Referral::query()
            ->where('commission_type', $commissionType)
            ->orderBy('level')
            ->get()
            ->keyBy('level');
    
        if ($tiers->isEmpty()) {
            return;
        }
    
        // Helper: required tokens per level
        $requiredTokens = function (int $level): int {
            if ($level <= 2)  return 20000;
            if ($level <= 5)  return 60000;
            return 120000; // 6–8 (and higher treated same as 6–8)
        };
    
        // Label for notify
        $typeLabel = $commissionType === 'deposit_commission'
            ? 'Deposit'
            : ($commissionType === 'interest_commission' ? 'Interest' : 'Invest');
    
        $transactions = [];
        $current      = $user;
        $maxLevel     = $tiers->keys()->max();
    
        DB::transaction(function () use (
            $amount, 
            $tiers, 
            $requiredTokens, 
            $typeLabel, 
            $trx, 
            &$transactions, 
            &$current, 
            $maxLevel
        ) {
    
            for ($level = 1; $level <= $maxLevel; $level++) {
                // Walk up 1 referrer each loop
                $referrer = $current->referrer ?? null;
                if (!$referrer) {
                    break;
                }
    
                $tier = $tiers->get($level);
                if (!$tier) {
                    // No configured tier for this level → stop
                    break;
                }
    
                // Unlock check using referrer's token balance
                $need = $requiredTokens($level);
                $has  = (int)($referrer->token_balance ?? 0);
                if ($has >= $need) {
                    // Calculate commission
                    $com = round(($amount * $tier->percent) / 100, 8); // precision safety
    
                    // Increment referral wallet and capture post balance
                    $referrer->referral_walltet = (float)($referrer->referral_walltet ?? 0) + $com;
                    $referrer->save();
    
                    $transactions[] = [
                        'user_id'      => $referrer->id,
                        'amount'       => $com,
                        'post_balance' => $referrer->referral_walltet, // correct wallet
                        'charge'       => 0,
                        'trx_type'     => '+',
                        'details'      => 'Level '.$level.' Referral Commission from '.$current->username,
                        'trx'          => $trx,
                        'wallet_type'  => 'referral_walltet',
                        'remark'       => 'referral_commission',
                        'created_at'   => now(),
                    ];
    
                    // Notify referrer (post_balance from referral_wallet)
                    notify($referrer, 'REFERRAL_COMMISSION', [
                        'amount'       => showAmount($com),
                        'post_balance' => showAmount($referrer->referral_walltet),
                        'trx'          => $trx,
                        'level'        => ordinal($level),
                        'type'         => $typeLabel,
                    ]);
                }
                // If not unlocked, we simply skip giving commission for this level and continue up the chain.
    
                // Move up the chain
                $current = $referrer;
            }
    
            if (!empty($transactions)) {
                \App\Models\Transaction::insert($transactions);
            }
        });
    }



    /**
     * Capital return
     *
     * @param object $invest
     * @param object $user
     * @return void
     */

    public static function capitalReturn($invest, $user)
    {
        $capital = $invest->amount;
        $user->return_walltet += $capital;
        $user->save();

        $invest->capital_back = 1;
        $invest->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $capital;
        $transaction->charge       = 0;
        $transaction->post_balance = $user->return_walltet;
        $transaction->trx_type     = '+';
        $transaction->trx          = getTrx();
        $transaction->wallet_type  = 'return_walltet';
        $transaction->remark       = 'capital_return';
        $transaction->details      = showAmount($capital) . ' ' . gs()->cur_text . ' capital back from ' . @$invest->plan->name;
        $transaction->save();
    }
    
    public static function canelPlan($invest, $user)
    {
        $general = gs();
         
        $charge      = ($invest->amount * $general->plan_cancel_p_charg) / 100;
        $capital = $invest->amount - $charge;
        $user->return_walltet += $capital;
        $user->save();

        $invest->capital_back = 1;
        $invest->status = 2;
        $invest->save();

        $transaction               = new Transaction();
        $transaction->user_id      = $user->id;
        $transaction->amount       = $capital;
        $transaction->charge       = $charge;
        $transaction->post_balance = $user->return_walltet;
        $transaction->trx_type     = '+';
        $transaction->trx          = getTrx();
        $transaction->wallet_type  = 'return_walltet';
        $transaction->remark       = 'capital_return';
        $transaction->details      = showAmount($capital) . ' ' . gs()->cur_text . ' capital back from ' . @$invest->plan->name;
        $transaction->save();
    }
}
